/*********************************************************
 * Copyright (C) 2000-2012 VMware, Inc. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation version 2 and no later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 *
 *********************************************************/

/*
 * vmballoon.h: Definitions and macros for vmballoon driver.
 */

#ifndef	VMBALLOON_H
#define	VMBALLOON_H

#include "balloonInt.h"
#include "vm_basic_types.h"
#include "dbllnklst.h"
#include "os.h"

/*
 * Page allocation flags
 */
typedef enum BalloonPageAllocType {
   BALLOON_PAGE_ALLOC_NOSLEEP = 0,
   BALLOON_PAGE_ALLOC_CANSLEEP = 1,
   BALLOON_PAGE_ALLOC_TYPES_NR,	// total number of alloc types
} BalloonPageAllocType;

/*
 * Types
 */

typedef struct {
   /* current status */
   uint32 nPages;
   uint32 nPagesTarget;

   /* adjustment rates */
   uint32 rateNoSleepAlloc;
   uint32 rateAlloc;
   uint32 rateFree;

   /* high-level operations */
   uint32 timer;

   /* primitives */
   uint32 primAlloc[BALLOON_PAGE_ALLOC_TYPES_NR];
   uint32 primAllocFail[BALLOON_PAGE_ALLOC_TYPES_NR];
   uint32 primFree;
   uint32 primErrorPageAlloc;
   uint32 primErrorPageFree;

   /* monitor operations */
   uint32 lock;
   uint32 lockFail;
   uint32 unlock;
   uint32 unlockFail;
   uint32 target;
   uint32 targetFail;
   uint32 start;
   uint32 startFail;
   uint32 guestType;
   uint32 guestTypeFail;
} BalloonStats;

#define BALLOON_ERROR_PAGES             16

typedef struct {
   PageHandle page[BALLOON_ERROR_PAGES];
   uint32 pageCount;
} BalloonErrorPages;

#define BALLOON_CHUNK_PAGES             1000

typedef struct BalloonChunk {
   PageHandle page[BALLOON_CHUNK_PAGES];
   uint32 pageCount;
   DblLnkLst_Links node;
} BalloonChunk;

struct BalloonOps;

typedef struct {
   /* sets of reserved physical pages */
   DblLnkLst_Links chunks;
   int nChunks;

   /* transient list of non-balloonable pages */
   BalloonErrorPages errors;

   BalloonGuest guestType;

   /* balloon size */
   int nPages;
   int nPagesTarget;

   /* reset flag */
   int resetFlag;

   /* adjustment rates (pages per second) */
   int rateAlloc;
   int rateFree;

   /* slowdown page allocations for next few cycles */
   int slowPageAllocationCycles;

   /* statistics */
   BalloonStats stats;

   /* hypervisor exposed capabilities */
   BalloonCapabilities hypervisorCapabilities;

   /* balloon operations, tied to the capabilities */
   const struct BalloonOps *balloonOps;

   /* Either the batch page handle, or the page to lock on v2 */
   PageHandle pageHandle;
   Mapping batchPageMapping;
   BalloonBatchPage *batchPage;
   uint16 batchMaxPages;

   BalloonChunk *fallbackChunk;
} Balloon;

typedef struct BalloonOps {
   void (*addPage)(Balloon *b, uint16 idx, PageHandle page);
   int (*lock)(Balloon *b, uint16 nPages, uint32 *target);
   int (*unlock)(Balloon *b, uint16 nPages, uint32 *target);
} BalloonOps;

/*
 * Operations
 */

Bool Balloon_Init(BalloonGuest guestType);
void Balloon_Cleanup(void);

void Balloon_QueryAndExecute(void);

const BalloonStats *Balloon_GetStats(void);

#endif	/* VMBALLOON_H */
